/******************************************************/
/* Examples Program For "ET-BASE dsPIC30F4011"  Board */
/* Hardware    : ET-BASE dsPIC30F4011                 */
/* Target MCU  : dsPIC30F4011                         */
/*       	   : X-TAL : 7.3728 MHz                   */
/*             : Run 117.9648MHz                      */
/*             : Selec OSC Mode = XT w/PLL 16x        */
/* Compiler    : MPLAB v8.40 + C30 v3.20B      	      */
/* Last Update : 12/October/2009                      */
/******************************************************/
/* Function    : Example Used Internal EEPROM         */
/*             : Write/Verify ---> UART1              */	
/******************************************************/
/* Accress Internal EEPROM 1 KByte Address Space      */
/* Address MAP : 0x7FFC00 - 0x7FFFFE                  */
/******************************************************/

#include <p30f4011.h>                                			// dsPIC30F4011 MPU Register
#include <stdio.h>												// Used "sprintf" Function
#include "uart.h"												// Used UART Config

/* Setup Configuration For ET-BASE dsPIC30F4011 */
_FOSC(CSW_FSCM_OFF & XT_PLL16);									// Disable Clock Switching,Enable Fail-Salf Clock
                                                            	// Clock Source = Primary XT + (PLL x 16)
_FWDT(WDT_OFF);													// Disable Watchdog 
_FBORPOR(PBOR_OFF & PWRT_64 & MCLR_EN);					    	// Disable Brown-Out ,Power ON = 64mS,Enable MCLR
_FGS(CODE_PROT_OFF);											// Code Protect OFF
/* End Configuration For ET-BASE dsPIC30F4011 */


char uart_buf[40]; 												// "sprintf" Buffer

const unsigned int eep_buff[31] =  {0x4142,0x4344,0x4546,		// AB CD EF
                              		0x4748,0x494A,0x4B4C,		// GH IJ KL
                              		0x4D4E,0x4F50,0x5152,		// MN OP QR
                              		0x5354,0x5556,0x5758,		// ST UV WX
                              		0x595A,0x6162,0x6364,		// YZ ab cd
						      		0x6566,0x6768,0x696A,		// ef gh ij
                              		0x6B6C,0x6D6E,0x6F70,		// kl mn op
                              		0x7172,0x7374,0x7576,		// qr st uv
                              		0x7778,0x797A,0x3031,		// wx yz 01
                              		0x3233,0x3435,0x3637,		// 23 45 67
                              		0x3839};					// 89

/* pototype  section */
void init_uart(void);											// Initial UART = 9600,N,8,1
void Erase_EEPROM(unsigned long address);						// Erase Word EEPROM 
void Write_EEPROM(unsigned long address,unsigned int data);		// Write Word EEPROM                  	
unsigned int Read_EEPROM(unsigned long address);				// Read  Word EEPROM                         	

int main(void)
{  
  unsigned int  count;											// Counter Read/Write 
  unsigned int  eep_data; 										// EEPROM Buffer Data
  unsigned long eep_addr;										// EEPROM Buffer Address

  init_uart();													// Initial UART = 9600,N,8,1   
  sprintf(uart_buf,"\n\rET-BASE dsPIC30F4011 : EEPROM\n\r");	// Print Message String 
  putsUART1((unsigned int *)uart_buf);                      	// Print uart_buff to UART1
  while(BusyUART1());											// Wait putsUART1 Complete
  
  sprintf(uart_buf,"Demo Accress Internal 1K EEPROM\n\n\r");	// Print Message String 
  putsUART1((unsigned int *)uart_buf);                      	// Print uart_buff to UART1
  while(BusyUART1());											// Wait putsUART1 Complete  

  sprintf(uart_buf,"Write[A..Z,a..z,0..9] to EEPROM Address[0x7FFC00]..."); 
  putsUART1((unsigned int *)uart_buf);                    		// Print uart_buff to UART1
  while(BusyUART1());											// Wait putsUART1 Complete

  // Test Write EEPROM 0x7FFC00,0x7FFC02,0x7FFC04.,,0x7FFFFE
  // Write Address 0x7FFC00 ---> A..Z,a..z,0..9
  eep_addr = 0x7FFC00;											// Start Address Write EEPROM  
  for (count = 0; count < 31; count++)							// Write 31 Word to EEPROM
  {     
    Erase_EEPROM(eep_addr);										// Erase 1 Word
    Write_EEPROM(eep_addr,eep_buff[count]);   					// Write Data 1 Word    
    eep_addr = eep_addr+2; 										// 0x7FFC00,0x7FFC02,0x7FFC04.,,0x7FFFFE
  }  

  sprintf(uart_buf,"Complete\n\r\r"); 
  putsUART1((unsigned int *)uart_buf);                    		// Print uart_buff to UART1
  while(BusyUART1());											// Wait putsUART1 Complete

  sprintf(uart_buf,"Read EEPROM Address[0x7FFC00].."); 
  putsUART1((unsigned int *)uart_buf);                    		// Print uart_buff to UART1
  while(BusyUART1());											// Wait putsUART1 Complete

  // Test Read EEPROM 0x7FFC00,0x7FFC02,0x7FFC04.,,0x7FFFFE
  eep_addr = 0x7FFC00;											// Start Address Write EEPROM
  for (count = 0; count < 31; count++)							// Read 31 Word From EEPROM
  {    
    eep_data = Read_EEPROM(eep_addr);    	 					// Read Data
    WriteUART1((eep_data&0xFF00)>>8);							// Print High Byte EEPROM Data to UART1
    while(BusyUART1());    										// Wait WriteUART1 Complete						
    WriteUART1((eep_data&0x00FF));								// Print Low Byte EEPROM Data to UART1
    while(BusyUART1());    										// Wait WriteUART1 Complete	
    eep_addr = eep_addr+2;  									// 0x7FFC00,0x7FFC02,0x7FFC04.,,0x7FFFFE    
  }  

  while(1);  													// Loop Continue      
}

/*********************************/
/* Initial UART for dsPIC30F4011 */
/* 9600,N,8,1 / 117.9648MHz Fosc */
/*********************************/
void init_uart()
{		  
  CloseUART1();												// Disable UART1 Before New Config

  // Config UART1 Interrupt Control
  ConfigIntUART1(UART_RX_INT_DIS &							// Disable RX Interrupt
    		     UART_RX_INT_PR2 &							// RX Interrupt Priority = 2
    		     UART_TX_INT_DIS &							// Disable TX Interrupt
    		     UART_TX_INT_PR3);							// TX Interrupt Priority = 3  
  // Open UART1 = Mode,Status,Baudrate              
  OpenUART1(UART_EN	&										// Enable UART(UART Mode)
            UART_IDLE_STOP &								// Disable UART in IDLE Mode 
 			UART_ALTRX_ALTTX & 								// Select U1TX=RC13,U1RX=RC14
            UART_DIS_WAKE &									// Disable Wake-Up
			UART_DIS_LOOPBACK &								// Disable Loop Back
			UART_DIS_ABAUD &								// Disable Auto Baudrate
  			UART_NO_PAR_8BIT &								// UART = 8 Bit, No Parity
 			UART_1STOPBIT,									// UART = 1 Stop Bit

	  		// Config UART1 Status
  			UART_INT_TX & 									// Select Interrupt After TX Complete
	 		UART_TX_PIN_NORMAL &							// Normal U1TX Mode
 			UART_TX_ENABLE &								// Enable U1TX
 	 		UART_INT_RX_BUF_FUL &							// Flasg Set After RX Complete 
  			UART_ADR_DETECT_DIS &              				// Disable Check Address 
			UART_RX_OVERRUN_CLEAR,							// Clear Overrun Flag

  			// ET-BASE dsPIC30F4011 Hardware Board
  			// XTAL = 7.3728MHz
  			// Fosc = 7.3728 MHz x 16 = 117.9648 MHz
  			// Fcy(UART) = Fosc / 4 
  			//           = 117.9648 / 4 = 29.4912 MHz
  			// U1BRG = [Fcy/(16xBaud)]-1
  			//       = [29.4912 MHz / (16x9600)] - 1
  			//       = 191 = BFH			
  			191);											// ET-BASE dsPIC30F4011 UART Baudrate = 9600 BPS
}

/********************************/
/* dsPIC30F4011 Internal EEPROM */
/* Internal EEPROM Write 1 Word */
/* Input : address(W1:W0)       */
/********************************/
void Erase_EEPROM(unsigned long address)					// Write Word EEPROM 
                  						
{  
  __asm__ volatile("MOV W1,NVMADRU");						// Upper Address (Address A[23..16])
  __asm__ volatile("MOV W0,NVMADR");                        // Lower Adress  (Address A[15..00])
  __asm__ volatile("MOV #0x4044,W0");						// Erase = 1 Word 
  __asm__ volatile("MOV W0,NVMCON");	
  __asm__ volatile("DISI #5");							    // Disable all Interrupt = 5 Instruction 
  __asm__ volatile("MOV #0x55,W0");							// Start Erase Command 
  __asm__ volatile("MOV W0,NVMKEY");
  __asm__ volatile("MOV #0xAA,W0");
  __asm__ volatile("MOV W0,NVMKEY");

  NVMCONbits.WR = 1;										// Start Erase Cycle (WR = 1)  
  while(NVMCONbits.WR);										// Wait WR = 0 (Erase Complete)  
  NVMCONbits.WREN = 0;										// Reset Startus  
}

/********************************/
/* dsPIC30F4011 Internal EEPROM */
/* Internal EEPROM Write 1 Word */
/* Input : address(W1:W0)       */
/*       : data(W2)             */
/********************************/
void Write_EEPROM(unsigned long address,unsigned int data)	// Write Word EEPROM 
                  						
{    
  __asm__ volatile("MOV W1,NVMADRU");						// Upper Address (Address A[23..16])
  __asm__ volatile("MOV W1,TBLPAG");						// Page  Address (Address A[23..16])
  __asm__ volatile("MOV W0,NVMADR");                        // Lower Adress  (Address A[15..00])   
  __asm__ volatile("TBLWTL W2,[W0]");						// Prepared Data For Program
  __asm__ volatile("MOV #0x4104,W0");						// Program = 1 Word 
  __asm__ volatile("MOV W0,NVMCON");	
  __asm__ volatile("DISI #5");							    // Disable all Interrupt = 5 Instruction 
  __asm__ volatile("MOV #0x55,W0");							// Start Erase Command 
  __asm__ volatile("MOV W0,NVMKEY");
  __asm__ volatile("MOV #0xAA,W0");
  __asm__ volatile("MOV W0,NVMKEY");

  NVMCONbits.WR = 1;										// Start Program Cycle (WR = 1)  
  while(NVMCONbits.WR);										// Wait WR = 0 (Program Complete)     
  NVMCONbits.WREN = 0;										// Reset Startus
}

/********************************/
/* dsPIC30F4011 Internal EEPROM */
/* Internal EEPROM Read 1 Word  */
/* Input  : address(W1:W0)      */
/* Output : data(W4)            */
/********************************/
unsigned int Read_EEPROM(unsigned long address)				// Read Word EEPROM  			
{
  __asm__ volatile("MOV W1,NVMADRU");						// Upper Address (Address A[23..16])
  __asm__ volatile("MOV W1,TBLPAG");						// Page  Address (Address A[23..16])
  __asm__ volatile("MOV W0,NVMADR");                        // Lower Adress  (Address A[15..00])  
  __asm__ volatile("TBLRDL [W0],W4");
  return (WREG4);											// Return Data Result
}

